﻿using System;
using System.Collections.Generic;
using System.Text;

namespace IndianHealthService.BMXNet
{

    /// <summary>
    /// Access to local services are access through this interface.
    /// </summary>
    public interface LocalEventService
    {
        /// <summary>
        /// Subscribe to the local event named anEventName.  
        /// </summary>
        /// <remarks>
        /// To receive published events, handle the ApplicationEvent.  If you explicity subscript
        /// to the "REFRESH" event then both RefreshRequested and ApplicationEvent will be triggers.       
        /// </remarks>
        /// <param name="anEventName">The name of the event.  This name should be shared and wellknown between many components.</param>
        void Subscribe(String anEventName);
        
        /// <summary>
        /// Checks to see there are any subscriptions for this event.  
        /// </summary>
        /// <remarks>
        /// If publishing an event, there is no need to first check if there are subscribers unless there is a
        /// performance hit for the application to Trigger the event.
        /// </remarks>
        /// <param name="anEventName">The name of the event checking to see if it's been subscribed already.</param>
        /// <returns></returns>
        bool HasSubscribers(String anEventName);


        /// <summary>
        /// Unscribe scribes from the event. It is not mandatory to unsubscribe from this event at the end of
        /// your object's lifecycle but you should either subsubscribe or remote the event handler.
        /// </summary>
        /// <param name="anEventName">The name of the event unsubscribing from.</param>
        /// <returns></returns>
        void Unsubscribe(String anEventName);

        /// <summary>
        /// This commonly broadcasted event is request to refresh content.  
        /// </summary>
        /// <remarks>
        /// In the EHR, this is equivalent to the "REFRESH" event.  The WinFramework also uses "REFRESH" as the
        /// event name for triggering a refresh request.
        /// </remarks>
        /// <example>
        /// The following code from Well Child Module's Growth Chart shows registering for the Refresh event
        /// at startup:
        /// <code>
        ///     public void Startup()
        ///     {         
        ///         this.LocalSession.EventServices.RefreshRequested += new EventHandler(Session_RefreshRequested);
        ///     }
        ///     
        ///     void Session_RefreshRequested(object sender, EventArgs e)
        ///     {
        ///         this.RefreshAll();
        ///     }
        ///</code>
        /// 
        /// </example>
        event EventHandler RefreshRequested;


        /// <summary>
        /// Local event subscribers will have this event triggered.  
        /// </summary>
        /// <remarks>
        /// The "REFRESH" event is filtered out and is triggered using the RefreshRequested event
        /// </remarks>
        /// <example>
        /// The Well Child Module growth chart displays Height and Weight.  When the EHR vitals component adds
        /// new Heights and Weights it publishes a LocalEvent with an EventType PCC*MSR.  If the growth chart detects
        /// the Vitals update event it then refreshes the growth chart:
        /// <code>
        ///     public void Startup()
        ///     {
        ///         this.LocalSession.EventServices.ApplicationEvent+= new EventHandler&lt;LocalEventArgs&gt;(Session_ApplicationEvent);
        ///     }
        ///     
        /// 
        ///  void Session_ApplicationEvent(object sender, LocalEventArgs e)
        ///  {
        ///     if (this.IsVitalsUpdatedEvent(e))
        ///     {
        ///         this.RefreshAll();
        ///     }
        ///  }
        ///  
        ///  protected bool IsVitalsUpdatedEvent(LocalEventArgs anEvent)
        ///  {
        ///     return anEvent.EventType.StartsWith("PCC") &amp;&amp; anEvent.EventType.EndsWith("MSR");
        ///  }
        /// </code>
        /// </example>
        event EventHandler<LocalEventArgs> ApplicationEvent;

        /// <summary>
        /// To publish an event call TriggerEvent with the event name and optional details.  There is no
        /// restriction on what local events that can be triggered.  A "REFRESH" event can be broadcast as
        /// well as any other application specific or well-known loal event.
        /// </summary>
        /// <param name="anEventName">The event name.  This is normally a well-known event that multiple components used for event-driven communications.</param>
        /// <param name="someDetails">An arbitrary string containing event specific data</param>
        void TriggerEvent(String anEventName, String someDetails);        
    }
}


